<?php

use App\Models\User;
use App\Models\User\UserEmployee;
use App\Models\User\UserProject;
use App\Models\Emp\UserEmployeeCvCode;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Livewire\Attributes\Layout;
use Livewire\Volt\Component;
use Livewire\WithFileUploads;
use Illuminate\View\View;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile as LivewireTemp;
use Illuminate\Support\Facades\Storage;

/**
 * ⚙️  Livewire Volt Register Component (full revision)
 */

new #[Layout('components.frontend.layouts.struct')] class extends Component {
    use WithFileUploads;

    // ───── Bindings ─────
    public int $currentStep = 1;
    public string $name_ar = '';
    public string $email = '';
    public array $phone = [''];
    public string $project_code = '';
    public string $national_id = '';
    public array $id_card_images = [];
    public array $id_card_images_buffer = [];
    public int|string|null $job_title_id = null;
    public array $experience_roads = [];
    public array $experience_airports = [];
    public string $cv_code = '';
    public $profile_picture;
    public array $skills = [];
    public string $hire_date = '';
    public bool $agreed = false;

    // ───── Title ─────
    public function rendering(View $view): void
    {
        $view->title(__('frontend.tittlewelcomepage') . ' | ' . __('frontend.register'));
    }

    // ───── Rules / messages ─────
    protected function rules(): array
    {
        return [
            'cv_code' => [
                'required',
                'exists:users_employees_cv_codes,code',
                function ($attribute, $value, $fail) {
                    $cvCode = UserEmployeeCvCode::where('code', $value)->first();

                    if ($cvCode) {
                        if ($cvCode->used) {
                            $fail(__('validation.cv_code_already_used'));
                        }

                        if (trim($cvCode->name) !== trim($this->name_ar)) {
                            $fail(__('validation.cv_code_name_mismatch'));
                        }
                    }
                },
            ],
            'hire_date' => ['required', 'date', 'before_or_equal:today'],
            'name_ar' => [
                'required',
                'string',
                'max:255',
                'regex:/^[\p{Arabic}\s]+$/u',
                function ($attribute, $value, $fail) {
                    if (count(preg_split('/\s+/u', trim($value))) < 3) {
                        $fail(__('validation.fullname_required_three_words_ar'));
                    }
                },
            ],

            'email' => ['nullable', 'string', 'lowercase', 'email', 'max:255', Rule::unique('users', 'email')],
            'phone' => ['required', 'array', 'min:1'],
            'phone.*' => ['required', 'string', 'regex:/^(\+?20|0020)?1?\d{9}$/'],
            'project_code' => ['required', 'exists:users_projects,code'],
            'national_id' => [
                'required',
                'digits:14',
                'unique:users_employees,national_id',
                function ($attribute, $value, $fail) {
                    if (!preg_match('/^[23][0-9]{13}$/', $value)) {
                        return $fail(__('validation.national_id_format_invalid'));
                    }

                    $century = match (substr($value, 0, 1)) {
                        '2' => 1900,
                        '3' => 2000,
                        default => null,
                    };

                    $year = (int) substr($value, 1, 2);
                    $month = (int) substr($value, 3, 2);
                    $day = (int) substr($value, 5, 2);

                    if (!checkdate($month, $day, $century + $year)) {
                        return $fail(__('validation.national_id_birthdate_invalid'));
                    }

                    $govCode = substr($value, 7, 2);
                    $validGovCodes = ['01', '02', '03', '04', '11', '12', '13', '14', '15', '16', '17', '18', '19', '21', '22', '23', '24', '25', '26', '27', '28', '29', '31', '32', '33', '34', '35', '88'];

                    if (!in_array($govCode, $validGovCodes)) {
                        return $fail(__('validation.national_id_governorate_invalid'));
                    }
                    // ✅ تحقق النوع (ذكر/أنثى)
                    $genderDigit = substr($value, 12, 1);
                    if (!is_numeric($genderDigit)) {
                        return $fail(__('validation.national_id_gender_invalid'));
                    }
                },
            ],
            'id_card_images' => ['required', 'array', 'min:1'],
            'id_card_images.*' => ['file', 'mimes:jpeg,jpg,png,pdf', 'max:4096'],
            'job_title_id' => ['required', 'exists:users_job_titles,id'],
            'profile_picture' => ['nullable', 'image', 'mimes:jpeg,jpg,png', 'max:2048'],
            'skills' => ['nullable', 'array'],
            'skills.*' => ['string', 'max:50'],

            'agreed' => ['accepted'],
        ];
    }

    protected array $messages = [];

    // ───── Register ─────
    public function register(): void
    {
        try {
            // 1️⃣ Validate
            $this->validate($this->rules(), $this->messages, $this->attributes());

            // 2️⃣ Transactional save
            DB::transaction(function () {
                $nextId = (UserEmployee::max('id') ?? 0) + 1;
                $employeeCode = 'EMP-' . str_pad((string) $nextId, 6, '0', STR_PAD_LEFT);
                while (UserEmployee::where('employee_code', $employeeCode)->exists()) {
                    $nextId++;
                    $employeeCode = 'EMP-' . str_pad((string) $nextId, 6, '0', STR_PAD_LEFT);
                }
                // dd($idCardPath, $this->id_card_image instanceof TemporaryUploadedFile, $this->id_card_image);
                $idCardPaths = [];
                $folder = 'employees/' . $employeeCode;
                foreach ($this->id_card_images as $file) {
                    if ($file instanceof LivewireTemp) {
                        $nameForFile = Str::slug($this->name_ar);
                        $randomCode = Str::random(6);
                        $extension = $file->getClientOriginalExtension();
                        $filename = "{$nameForFile}_{$randomCode}.{$extension}";

                        // تخزين داخل مجلد مخصص لكل موظف
                        $idCardPaths[] = $file->storeAs($folder, $filename, 'public');
                    }
                }

                $info = $this->parseNationalId($this->national_id);

                // رفع الصورة الرسمية
                $profilePicturePath = null;
                if ($this->profile_picture instanceof LivewireTemp) {
                    $nameForFile = Str::slug($this->name_ar);
                    $randomCode = Str::random(6);
                    $extension = $this->profile_picture->getClientOriginalExtension();
                    $filename = "{$nameForFile}_profile_{$randomCode}.{$extension}";

                    $profilePicturePath = $this->profile_picture->storeAs($folder, $filename, 'public');
                }

                $employee = UserEmployee::create([
                    'project_code' => $this->project_code,
                    'job_title_id' => $this->job_title_id,
                    'employee_code' => $employeeCode,
                    'full_name' => ['ar' => $this->name_ar],
                    'phone' => json_encode($this->phone),
                    'national_id' => $this->national_id,
                    'birth_date' => $info['birth_date'],
                    'governorate' => $info['governorate'],
                    'gender' => $info['gender'],
                    'id_card_image' => json_encode($idCardPaths),
                    'hiring_date' => $this->hire_date,
                    'cv' => json_encode(
                        array_filter(
                            [
                                'work_experience' => [
                                    'roads' => $this->experience_roads,
                                    'airports' => $this->experience_airports,
                                ],
                                'email' => $this->email,
                                'profile_picture' => $profilePicturePath ?? null,
                                'skills' => $this->skills,
                            ],
                            fn($v) => !is_null($v) && $v !== '',
                        )
                        , JSON_UNESCAPED_UNICODE),
                ]);

                UserEmployeeCvCode::where('code', $this->cv_code)->update([
                    'used' => true,
                    'updated_at' => now(),
                ]);
            });

            // 3️⃣ Success toast
            session()->flash('toast', [
                'type' => 'success',
                'msg' => __('frontend.cv_saved_successfully'),
                'duration' => 5000,
            ]);

            // $this->redirectIntended(route('home', absolute: false), navigate: true);
            $this->redirect(route('home'), navigate: true);
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('notify', type: 'error', message: $e->validator->errors()->first(), duration: 6000);
            // foreach ($e->validator->errors()->all() as $err) {
            //     $this->dispatch('notify', type: 'error', message: $err, duration: 6000);
            // }
        } catch (\Throwable $e) {
            $this->dispatch('notify', type: 'error', message: __('frontend.anothererror'), duration: 6000);
            // report($e);
            //Check Full Error
            $this->dispatch('notify', type: 'error', message: $e->getMessage(), duration: 10000);
        }
    }
    private function parseNationalId(string $id): array
    {
        // 🗓️ تاريخ الميلاد
        $century = match (substr($id, 0, 1)) {
            '2' => 1900,
            '3' => 2000,
            default => null,
        };

        $year = (int) substr($id, 1, 2);
        $month = (int) substr($id, 3, 2);
        $day = (int) substr($id, 5, 2);

        $birthDate = $century ? sprintf('%04d-%02d-%02d', $century + $year, $month, $day) : null;

        // 🏙️ المحافظة
        $govCode = substr($id, 7, 2);
        $governorates = [
            '01' => ['ar' => 'القاهرة', 'en' => 'Cairo'],
            '02' => ['ar' => 'الإسكندرية', 'en' => 'Alexandria'],
            '03' => ['ar' => 'بورسعيد', 'en' => 'Port Said'],
            '04' => ['ar' => 'السويس', 'en' => 'Suez'],
            '11' => ['ar' => 'دمياط', 'en' => 'Damietta'],
            '12' => ['ar' => 'الدقهلية', 'en' => 'Dakahlia'],
            '13' => ['ar' => 'الشرقية', 'en' => 'Sharqia'],
            '14' => ['ar' => 'القليوبية', 'en' => 'Qalyubia'],
            '15' => ['ar' => 'كفر الشيخ', 'en' => 'Kafr El-Sheikh'],
            '16' => ['ar' => 'الغربية', 'en' => 'Gharbia'],
            '17' => ['ar' => 'المنوفية', 'en' => 'Monufia'],
            '18' => ['ar' => 'البحيرة', 'en' => 'Beheira'],
            '19' => ['ar' => 'الإسماعيلية', 'en' => 'Ismailia'],
            '21' => ['ar' => 'الجيزة', 'en' => 'Giza'],
            '22' => ['ar' => 'بني سويف', 'en' => 'Beni Suef'],
            '23' => ['ar' => 'الفيوم', 'en' => 'Faiyum'],
            '24' => ['ar' => 'المنيا', 'en' => 'Minya'],
            '25' => ['ar' => 'أسيوط', 'en' => 'Assiut'],
            '26' => ['ar' => 'سوهاج', 'en' => 'Sohag'],
            '27' => ['ar' => 'قنا', 'en' => 'Qena'],
            '28' => ['ar' => 'أسوان', 'en' => 'Aswan'],
            '29' => ['ar' => 'الأقصر', 'en' => 'Luxor'],
            '31' => ['ar' => 'البحر الأحمر', 'en' => 'Red Sea'],
            '32' => ['ar' => 'الوادي الجديد', 'en' => 'New Valley'],
            '33' => ['ar' => 'مطروح', 'en' => 'Matrouh'],
            '34' => ['ar' => 'شمال سيناء', 'en' => 'North Sinai'],
            '35' => ['ar' => 'جنوب سيناء', 'en' => 'South Sinai'],
            '88' => ['ar' => 'خارج الجمهورية', 'en' => 'Outside Egypt'],
        ];

        $governorate = [
            'code' => $govCode,
            'en' => $governorates[$govCode]['en'] ?? null,
            'ar' => $governorates[$govCode]['ar'] ?? null,
        ];

        // 👤 النوع (رقم فردي = ذكر / زوجي = أنثى)
        $genderDigit = (int) substr($id, 12, 1);
        $gender = null;
        if (strlen($id) >= 13) {
            $genderDigit = (int) substr($id, 12, 1);
            $gender = $genderDigit % 2 === 0 ? 'female' : 'male';
        }

        return [
            'birth_date' => $birthDate,
            'governorate' => $governorate,
            'gender' => $gender,
        ];
    }

    public function updatedIdCardImages($newFiles)
    {
        // Merge القيمة المؤقتة مع الملفات الجديدة
        $merged = collect($this->id_card_images_buffer)
            ->merge($newFiles)
            ->unique(function ($file) {
                return $file->getClientOriginalName() . $file->getSize();
            })
            ->values()
            ->all();

        $this->id_card_images = $merged;
        $this->id_card_images_buffer = $merged; // تحديث النسخة المؤقتة
    }

    public function removeImage($index)
    {
        unset($this->id_card_images[$index]);
        $this->id_card_images = array_values($this->id_card_images);

        unset($this->id_card_images_buffer[$index]);
        $this->id_card_images_buffer = array_values($this->id_card_images_buffer);
    }

    protected function attributes(): array
    {
        return [
            'cv_code' => __('frontend.cv_code'),
            'hire_date' => __('frontend.hire_date'),
            'name_ar' => __('frontend.fullname'),
            'email' => __('frontend.emailadress'),
            'phone' => __('frontend.phone'),
            'phone.*' => __('frontend.phone'),
            'project_code' => __('frontend.project'),
            'national_id' => __('frontend.national_id'),
            'id_card_images' => __('frontend.attach_documents'),
            'job_title_id' => __('frontend.job_title'),
            'agreed' => __('frontend.agree'),
        ];
    }

    public function nextStep()
    {
        try {
            $this->validate($this->stepRules(), $this->messages, $this->attributes());
            $this->currentStep++;
        } catch (\Illuminate\Validation\ValidationException $e) {
            // نعرض أول خطأ فقط
            $firstError = $e->validator->errors()->first();
            $this->dispatch('notify', type: 'error', message: $firstError, duration: 6000);
        }
    }

    public function previousStep()
    {
        $this->currentStep--;
    }

    private function stepRules(): array
    {
        if ($this->currentStep === 1) {
            return [
                'cv_code' => [
                    'required',
                    'exists:users_employees_cv_codes,code',
                    function ($attribute, $value, $fail) {
                        $cvCode = UserEmployeeCvCode::where('code', $value)->first();

                        if ($cvCode) {
                            if ($cvCode->used) {
                                $fail(__('validation.cv_code_already_used'));
                            }

                            if (trim($cvCode->name) !== trim($this->name_ar)) {
                                $fail(__('validation.cv_code_name_mismatch'));
                            }
                        }
                    },
                ],
                'name_ar' => [
                    'required',
                    'string',
                    'max:255',
                    'regex:/^[\p{Arabic}\s]+$/u',
                    function ($attribute, $value, $fail) {
                        if (count(preg_split('/\s+/u', trim($value))) < 3) {
                            $fail(__('validation.fullname_required_three_words_ar'));
                        }
                    },
                ],
            ];
        }

        if ($this->currentStep === 2) {
            return [
                'email' => ['nullable', 'string', 'lowercase', 'email', 'max:255', Rule::unique('users', 'email')],
                'phone' => ['required', 'array', 'min:1'],
                'phone.*' => ['required', 'string', 'regex:/^(\+?20|0020)?1?\d{9}$/'],
            ];
        }

        if ($this->currentStep === 3) {
            return [
                'national_id' => [
                    'required',
                    'digits:14',
                    'unique:users_employees,national_id',
                    function ($attribute, $value, $fail) {
                        if (!preg_match('/^[23][0-9]{13}$/', $value)) {
                            return $fail(__('validation.national_id_format_invalid'));
                        }

                        $century = match (substr($value, 0, 1)) {
                            '2' => 1900,
                            '3' => 2000,
                            default => null,
                        };

                        $year = (int) substr($value, 1, 2);
                        $month = (int) substr($value, 3, 2);
                        $day = (int) substr($value, 5, 2);

                        if (!checkdate($month, $day, $century + $year)) {
                            return $fail(__('validation.national_id_birthdate_invalid'));
                        }

                        $govCode = substr($value, 7, 2);
                        $validGovCodes = ['01', '02', '03', '04', '11', '12', '13', '14', '15', '16', '17', '18', '19', '21', '22', '23', '24', '25', '26', '27', '28', '29', '31', '32', '33', '34', '35', '88'];

                        if (!in_array($govCode, $validGovCodes)) {
                            return $fail(__('validation.national_id_governorate_invalid'));
                        }

                        $genderDigit = substr($value, 12, 1);
                        if (!is_numeric($genderDigit)) {
                            return $fail(__('validation.national_id_gender_invalid'));
                        }
                    },
                ],
                'project_code' => ['required', 'exists:users_projects,code'],
                'hire_date' => ['required', 'date', 'before_or_equal:today'],
                'job_title_id' => ['required', 'exists:users_job_titles,id'],
            ];
        }

        if ($this->currentStep === 4) {
            return [
                'id_card_images' => ['required', 'array', 'min:1'],
                'id_card_images.*' => ['file', 'mimes:jpeg,jpg,png,pdf', 'max:4096'],
                'profile_picture' => ['nullable', 'image', 'mimes:jpeg,jpg,png', 'max:2048'],
                'skills' => ['nullable', 'array'],
                'skills.*' => ['string', 'max:50'],
                'agreed' => ['accepted'],
            ];
        }

        return [];
    }
};
?>
<!-- ───────────────────────────── View ───────────────────────────── -->
<div x-data="{ showTerms: false }"
    class="min-h-screen flex items-center justify-center bg-transparent transition duration-700 overflow-hidden relative">

    <!-- Loader عام للفورم -->
    <div wire:loading.delay
        class="absolute inset-0 bg-white/50 dark:bg-zinc-900/60 backdrop-blur-sm flex items-center justify-center z-40">
        <svg class="w-10 h-10 animate-spin" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"
            stroke="currentColor" stroke-width="2">
            <path stroke-linecap="round" stroke-linejoin="round"
                d="M12 4v1m6.364 1.636l-.707.707M20 12h-1m-1.636 6.364l-.707-.707M12 20v-1m-6.364-1.636l.707-.707M4 12h1m1.636-6.364l.707.707" />
        </svg>
    </div>

    <!-- Register form wrapper -->
    <div
        class="z-10 max-w-md w-full space-y-6 bg-white/80 dark:bg-zinc-900/80 backdrop-blur-md p-8 rounded-xl shadow-2xl border border-zinc-300 dark:border-zinc-700">
        <div class="flex flex-col items-center gap-2 animate-fade-in-down">
            <img src="/assets/img/logo.png" alt="Logo" class="w-[300px] h-auto mb-3 drop-shadow-lg" />
            <h2 class="text-2xl font-semibold text-zinc-800 dark:text-white">{{ __('frontend.add_cv') }}</h2>
        </div>

        <form wire:submit.prevent="register"
            @keydown.enter.prevent="
        if ($wire.currentStep < 4) {
            $wire.nextStep();
        }
    "
            class="space-y-3 animate-fade-in-down">


            <!-- Step Section with Pulse -->
            <div class="mb-6 space-y-2">

                <!-- Step Label -->
                <div class="flex justify-center items-center mb-2">
                    <span class="text-sm font-semibold text-zinc-700 dark:text-zinc-300">
                        {{ __('frontend.step') }} {{ $currentStep }} {{ __('frontend.of') }} 4
                    </span>
                </div>

                <!-- Steps Indicators -->
                <div class="flex justify-center items-center space-x-2 rtl:space-x-reverse mb-2">
                    <template x-for="step in 4" :key="step">
                        <div class="flex items-center justify-center w-8 h-8 rounded-full border-2 font-semibold transition-all duration-300 text-sm relative"
                            :class="{
                                'bg-indigo-600 text-white border-indigo-600 shadow-lg animate-pulse-step': $wire
                                    .currentStep === step,
                                'bg-white text-zinc-700 dark:bg-zinc-800 dark:text-zinc-300 border-zinc-300 dark:border-zinc-600': $wire
                                    .currentStep !== step
                            }">
                            <span x-text="step"></span>
                        </div>
                    </template>
                </div>

                <!-- Progress Bar -->
                <div class="w-full bg-zinc-200 rounded-full h-2.5 dark:bg-zinc-700">
                    <div class="bg-indigo-600 h-2.5 rounded-full transition-all duration-500"
                        :style="{ width: (($wire.currentStep - 1) / (4 - 1)) * 100 + '%' }">
                    </div>
                </div>

            </div>



            <div x-show="$wire.currentStep === 1">
                <!-- CV Code Input Box -->
                <div class="mb-6">
                    <label for="cv_code" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-2">
                        {{ __('frontend.enter_cv_code') }}
                    </label>

                    <div
                        class="flex rounded-lg overflow-hidden border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 focus-within:ring-2 focus-within:ring-indigo-500 transition-all duration-300">
                        <div
                            class="flex items-center px-3 bg-zinc-100 dark:bg-zinc-700 border-r border-zinc-300 dark:border-zinc-600 text-indigo-500 dark:text-indigo-400">
                            <!-- Icon -->
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" stroke-width="2"
                                viewBox="0 0 24 24" stroke-linecap="round" stroke-linejoin="round">
                                <line x1="4" y1="9" x2="20" y2="9"></line>
                                <line x1="4" y1="15" x2="20" y2="15"></line>
                                <line x1="10" y1="3" x2="8" y2="21"></line>
                                <line x1="16" y1="3" x2="14" y2="21"></line>
                            </svg>

                        </div>

                        <input type="text" id="cv_code" required
                            placeholder="{{ __('frontend.enter_cv_code_placeholder') }}"
                            class="flex-1 px-4 py-2 text-zinc-900 dark:text-white bg-transparent outline-none"
                            wire:model.defer="cv_code" />
                    </div>

                    <p class="mt-1 text-xs text-zinc-500 dark:text-zinc-400">
                        {{ __('frontend.enter_cv_code_hint') }}
                    </p>
                </div>


                <!-- Full name AR -->
                <div>
                    <label for="name_ar" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                        {{ __('frontend.fullname') }} (عربي)
                    </label>
                    <input type="text" id="name_ar" required placeholder="مايكل"
                        class="block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 focus:ring-indigo-500 outline-none transition"
                        wire:model.defer="name_ar" />
                </div>

                <div class="flex justify-between mt-6 space-x-2 rtl:space-x-reverse">
                    <!-- Previous Button -->
                    <button type="button" wire:click="previousStep" x-show="$wire.currentStep > 1"
                        x-transition.opacity.duration.300ms x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-gray-200 dark:bg-zinc-700 text-gray-700 dark:text-white font-medium rounded-lg hover:bg-gray-300 dark:hover:bg-zinc-600 transition-all duration-300">
                        ← {{ __('frontend.previous') }}
                    </button>

                    <!-- Next Button -->
                    <button type="button" wire:click="nextStep" x-show="$wire.currentStep < 4"
                        x-transition.opacity.duration.300ms x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-indigo-600 text-white font-medium rounded-lg hover:bg-indigo-700 transition-all duration-300">
                        {{ __('frontend.next') }} →
                    </button>

                    <!-- Submit Button -->
                    <button type="submit" x-show="$wire.currentStep === 4" x-transition.opacity.duration.300ms
                        x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-green-600 text-white font-medium rounded-lg hover:bg-green-700 transition-all duration-300">
                        🚀 {{ __('frontend.submit') }}
                    </button>
                </div>


            </div>

            <div x-show="$wire.currentStep === 2">
                <!-- Email -->
                <div>
                    <label for="email" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                        {{ __('frontend.emailadress') }}
                        <span class="text-xs text-zinc-500 dark:text-zinc-400">({{ __('frontend.optional') }})</span>
                    </label>
                    <div
                        class="flex rounded-lg overflow-hidden border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 focus-within:ring-2 focus-within:ring-indigo-500 transition-all duration-300">
                        <div
                            class="flex items-center px-3 bg-zinc-100 dark:bg-zinc-700 border-r border-zinc-300 dark:border-zinc-600 text-indigo-500 dark:text-indigo-400">
                            <!-- email icon -->
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" stroke-width="2"
                                viewBox="0 0 24 24" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M4 4h16v16H4z" stroke="none" />
                                <polyline points="22,6 12,13 2,6" />
                            </svg>
                        </div>
                        <input type="email" id="email" placeholder="example@email.com"
                            class="flex-1 px-4 py-2 text-zinc-900 dark:text-white bg-transparent outline-none"
                            wire:model.defer="email" />
                    </div>
                </div>

                <!-- Phone Number -->
                <div x-data="{
                    phones: @entangle('phone'),
                    addPhone() {
                        this.phones.push('');
                    },
                    removePhone(index) {
                        if (this.phones.length > 1) {
                            this.phones.splice(index, 1);
                        }
                    }
                }">
                    <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                        {{ __('frontend.phone') }}
                    </label>

                    <template x-for="(phone, index) in phones" :key="index">
                        <div class="flex items-center gap-2 mb-2">
                            <div
                                class="flex items-center gap-2 px-3 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 flex-1">
                                <!-- Icon + Prefix -->
                                <svg class="w-5 h-5 text-indigo-500 dark:text-indigo-400" fill="none"
                                    stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"
                                    stroke-linecap="round" stroke-linejoin="round">
                                    <path
                                        d="M22 16.92V21a2 2 0 0 1-2.18 2A19.79 19.79 0 0 1 3 5.18 2 2 0 0 1 5 3h4.09a1 1 0 0 1 1 .75l1.21 4.84a1 1 0 0 1-.29 1L9.1 11.1a16 16 0 0 0 6.9 6.9l1.51-1.51a1 1 0 0 1 1-.29l4.84 1.21a1 1 0 0 1 .75 1z" />
                                </svg>
                                <span class="text-sm font-medium select-none">+20</span>
                                <input type="tel" :id="'phone_' + index" placeholder="10xxxxxxxx"
                                    class="flex-1 px-2 py-1 bg-transparent outline-none text-zinc-900 dark:text-white"
                                    x-model="phones[index]" />
                            </div>

                            <!-- زر الحذف -->
                            <button type="button" @click="removePhone(index)"
                                class="text-red-500 hover:text-red-700 text-lg font-bold px-2 py-1">
                                &times;
                            </button>
                        </div>
                    </template>

                    <!-- زر إضافة رقم -->
                    <div class="flex justify-center mt-2">
                        <button type="button" @click="addPhone()"
                            class="px-3 py-1 bg-indigo-600 text-white text-sm rounded hover:bg-indigo-700 transition">
                            + {{ __('frontend.add_phone') }}
                        </button>
                    </div>

                    <p class="mt-1 text-xs text-zinc-500 dark:text-zinc-400">
                        {{ __('frontend.phone_hint') }}
                    </p>
                </div>

                <div class="flex justify-between mt-6 space-x-2 rtl:space-x-reverse">
                    <!-- Previous Button -->
                    <button type="button" wire:click="previousStep" x-show="$wire.currentStep > 1"
                        x-transition.opacity.duration.300ms x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-gray-200 dark:bg-zinc-700 text-gray-700 dark:text-white font-medium rounded-lg hover:bg-gray-300 dark:hover:bg-zinc-600 transition-all duration-300">
                        ← {{ __('frontend.previous') }}
                    </button>

                    <!-- Next Button -->
                    <button type="button" wire:click="nextStep" x-show="$wire.currentStep < 4"
                        x-transition.opacity.duration.300ms x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-indigo-600 text-white font-medium rounded-lg hover:bg-indigo-700 transition-all duration-300">
                        {{ __('frontend.next') }} →
                    </button>

                    <!-- Submit Button -->
                    <button type="submit" x-show="$wire.currentStep === 4" x-transition.opacity.duration.300ms
                        x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-green-600 text-white font-medium rounded-lg hover:bg-green-700 transition-all duration-300">
                        🚀 {{ __('frontend.submit') }}
                    </button>
                </div>
            </div>

            <div x-show="$wire.currentStep === 3">
                <!-- National ID -->
                <div>
                    <label for="national_id" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                        {{ __('frontend.national_id') }}
                    </label>
                    <div
                        class="flex rounded-lg overflow-hidden border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 focus-within:ring-2 focus-within:ring-indigo-500 transition-all duration-300">
                        <div
                            class="flex items-center px-3 bg-zinc-100 dark:bg-zinc-700 border-r border-zinc-300 dark:border-zinc-600 text-indigo-500 dark:text-indigo-400">
                            <!-- id icon -->
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" stroke-width="2"
                                viewBox="0 0 24 24" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M16 21v-2a4 4 0 00-8 0v2" />
                                <circle cx="12" cy="7" r="4" />
                                <rect x="2" y="3" width="20" height="18" rx="2" ry="2" />
                            </svg>
                        </div>
                        <input type="text" id="national_id" maxlength="14" required placeholder="xxxxxxxxxxxxxx"
                            class="flex-1 px-4 py-2 text-zinc-900 dark:text-white bg-transparent outline-none"
                            wire:model.defer="national_id" />
                    </div>
                </div>
                <!-- Project -->
                <div>
                    <label for="project_code" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                        {{ __('frontend.project') }}
                    </label>
                    <div
                        class="flex rounded-lg overflow-hidden border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 focus-within:ring-2 focus-within:ring-indigo-500 transition-all duration-300">
                        <div
                            class="flex items-center px-3 bg-zinc-100 dark:bg-zinc-700 border-r border-zinc-300 dark:border-zinc-600 text-indigo-500 dark:text-indigo-400">
                            <!-- project icon -->
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" stroke-width="2"
                                viewBox="0 0 24 24" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M21 16V8a2 2 0 0 0-2-2H5a2 2 0 0 0-2 2v8" />
                                <rect x="7" y="12" width="3" height="6" />
                                <rect x="14" y="12" width="3" height="6" />
                                <path d="M12 12V4" />
                            </svg>
                        </div>
                        <select id="project_code" wire:model.defer="project_code" required
                            class="flex-1 px-4 py-2 bg-transparent text-zinc-900 dark:text-white outline-none appearance-none">
                            <option value="">{{ __('frontend.chooseproject') }}</option>
                            @foreach (UserProject::all() as $project)
                                <option value="{{ $project->code }}">{{ $project->name }}</option>
                            @endforeach
                        </select>
                    </div>
                </div>

                <!-- Hire Date -->
                <div>
                    <label for="hire_date" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                        {{ __('frontend.hire_date') }}
                    </label>
                    <input type="date" id="hire_date" required
                        class="block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 focus:ring-indigo-500 outline-none transition"
                        wire:model.defer="hire_date" />
                </div>

                <!-- Job Title -->
                <div>
                    <label for="job_title_id" class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                        {{ __('frontend.job_title') }}
                    </label>
                    <div
                        class="flex rounded-lg overflow-hidden border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 focus-within:ring-2 focus-within:ring-indigo-500 transition-all duration-300">
                        <div
                            class="flex items-center px-3 bg-zinc-100 dark:bg-zinc-700 border-r border-zinc-300 dark:border-zinc-600 text-indigo-500 dark:text-indigo-400">
                            <!-- briefcase icon -->
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" stroke-width="2"
                                viewBox="0 0 24 24" stroke-linecap="round" stroke-linejoin="round">
                                <rect x="2" y="7" width="20" height="14" rx="2" ry="2">
                                </rect>
                                <path d="M16 3h-8a2 2 0 0 0-2 2v2h12V5a2 2 0 0 0-2-2z"></path>
                            </svg>
                        </div>
                        <select id="job_title_id" wire:model.defer="job_title_id" required
                            class="flex-1 px-4 py-2 bg-transparent text-zinc-900 dark:text-white outline-none appearance-none">
                            <option value="">{{ __('frontend.choosejobtitle') }}</option>
                            @foreach (App\Models\User\UserJobTitle::whereNotIn('id', [1, 2, 3])->get() as $jobTitle)
                                <option value="{{ $jobTitle->id }}">
                                    {{ app()->getLocale() == 'ar' ? $jobTitle->title_ar : $jobTitle->title_en }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                </div>

                <div class="flex justify-between mt-6 space-x-2 rtl:space-x-reverse">
                    <!-- Previous Button -->
                    <button type="button" wire:click="previousStep" x-show="$wire.currentStep > 1"
                        x-transition.opacity.duration.300ms x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-gray-200 dark:bg-zinc-700 text-gray-700 dark:text-white font-medium rounded-lg hover:bg-gray-300 dark:hover:bg-zinc-600 transition-all duration-300">
                        ← {{ __('frontend.previous') }}
                    </button>

                    <!-- Next Button -->
                    <button type="button" wire:click="nextStep" x-show="$wire.currentStep < 4"
                        x-transition.opacity.duration.300ms x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-indigo-600 text-white font-medium rounded-lg hover:bg-indigo-700 transition-all duration-300">
                        {{ __('frontend.next') }} →
                    </button>

                    <!-- Submit Button -->
                    <button type="submit" x-show="$wire.currentStep === 4" x-transition.opacity.duration.300ms
                        x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-green-600 text-white font-medium rounded-lg hover:bg-green-700 transition-all duration-300">
                        🚀 {{ __('frontend.submit') }}
                    </button>
                </div>
            </div>
            <div x-show="$wire.currentStep === 4">

                <h2 class="text-2xl font-bold text-zinc-800 dark:text-white mb-6 text-center">
                    📝 {{ __('frontend.cv_section_title') }} <!-- ترجم في lang/ar -->
                </h2>

                <!-- Profile Picture Upload -->
                <div class="mb-6" x-data>
                    <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-2">
                        {{ __('frontend.upload_profile_picture') }}
                    </label>

                    <div class="flex flex-col items-center justify-center border-2 border-dashed border-zinc-300 dark:border-zinc-600 rounded-lg p-4 cursor-pointer hover:bg-zinc-50 dark:hover:bg-zinc-800 transition"
                        @click="$refs.profilePicture.click()">

                        <svg class="w-10 h-10 text-indigo-500 mb-2" fill="none" stroke="currentColor"
                            stroke-width="2" viewBox="0 0 24 24" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M12 5v14M5 12h14" />
                        </svg>
                        <span class="text-sm text-zinc-500 dark:text-zinc-400">
                            {{ __('frontend.click_to_upload_picture') }}
                        </span>

                        <input type="file" x-ref="profilePicture" wire:model.live="profile_picture"
                            class="hidden" accept="image/jpeg,image/png,image/jpg" />
                    </div>

                    @if (!empty($profile_picture))
                        <div class="mt-3">
                            <img src="{{ $profile_picture->temporaryUrl() }}"
                                class="w-24 h-24 object-cover rounded-full border border-zinc-300 dark:border-zinc-600 mx-auto" />
                        </div>
                    @endif
                </div>


                <!-- Work Experience Section -->
                <div x-data="{
                    activeTab: 'roads',
                    experience_roads: @entangle('experience_roads'),
                    experience_airports: @entangle('experience_airports'),
                    addProject(tab) {
                        if (tab === 'roads') {
                            this.experience_roads = [...this.experience_roads, {
                                name: '',
                                start_date: '',
                                end_date: '',
                                description: ''
                            }];
                        } else {
                            this.experience_airports = [...this.experience_airports, {
                                name: '',
                                start_date: '',
                                end_date: '',
                                description: ''
                            }];
                        }
                    },

                    removeProject(tab, index) {
                        if (tab === 'roads') {
                            this.experience_roads.splice(index, 1);
                        } else {
                            this.experience_airports.splice(index, 1);
                        }
                    }

                }" class="space-y-4">

                    <h3 class="text-lg font-semibold text-zinc-700 dark:text-zinc-300 mb-2">
                        {{ __('frontend.work_experience') }}
                    </h3>

                    <!-- Tabs -->
                    <div class="flex space-x-2 mb-4">
                        <button type="button" @click="activeTab = 'roads'"
                            :class="activeTab === 'roads' ? 'bg-indigo-600 text-white' :
                                'bg-zinc-200 dark:bg-zinc-700 text-zinc-700 dark:text-zinc-300'"
                            class="px-4 py-2 rounded-lg font-medium transition">
                            {{ __('frontend.roads') }}
                        </button>
                        <button type="button" @click="activeTab = 'airports'"
                            :class="activeTab === 'airports' ? 'bg-indigo-600 text-white' :
                                'bg-zinc-200 dark:bg-zinc-700 text-zinc-700 dark:text-zinc-300'"
                            class="px-4 py-2 rounded-lg font-medium transition">
                            {{ __('frontend.airports') }}
                        </button>
                    </div>

                    <!-- Projects List -->
                    <template
                        x-for="(project, index) in activeTab === 'roads' ? experience_roads : experience_airports"
                        :key="index">
                        <div
                            class="space-y-2 p-4 border border-zinc-300 dark:border-zinc-600 rounded-lg bg-white dark:bg-zinc-800 relative">

                            <!-- Remove button -->
                            <button type="button" @click="removeProject(activeTab, index)"
                                class="absolute top-1 ltr:right-1 rtl:left-1 bg-red-600 hover:bg-red-700 text-white text-xs w-5 h-5 rounded-full shadow-md z-10 transition-transform transform hover:scale-110">
                                ×
                            </button>


                            <!-- Project Name -->
                            <div>
                                <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                                    {{ __('frontend.project_name') }}
                                </label>
                                <input type="text" x-model="project.name"
                                    class="block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 focus:ring-indigo-500 outline-none transition"
                                    placeholder="{{ __('frontend.project_name_placeholder') }}" />
                            </div>

                            <!-- Start Date -->
                            <div>
                                <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                                    {{ __('frontend.start_date') }}
                                </label>
                                <input type="date" x-model="project.start_date"
                                    class="block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 focus:ring-indigo-500 outline-none transition" />
                            </div>

                            <!-- End Date -->
                            <div>
                                <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                                    {{ __('frontend.end_date') }}
                                </label>
                                <input type="date" x-model="project.end_date"
                                    class="block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 focus:ring-indigo-500 outline-none transition" />
                            </div>

                            <!-- Project Description -->
                            <div>
                                <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                                    {{ __('frontend.project_description') }}
                                </label>
                                <textarea x-model="project.description"
                                    class="block w-full px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 focus:ring-indigo-500 outline-none transition"
                                    placeholder="{{ __('frontend.project_description_placeholder') }}"></textarea>
                            </div>

                        </div>
                    </template>

                    <!-- Add Project Button -->
                    <div class="flex justify-center mt-4">
                        <button type="button" @click="addProject(activeTab)"
                            class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition">
                            {{ __('frontend.add_project') }}
                        </button>
                    </div>
                </div>

                <!-- Skills Input -->
                <div x-data="{
                    newSkill: '',
                    skills: @entangle('skills'),
                    addSkill() {
                        if (this.newSkill.trim() !== '' && !this.skills.includes(this.newSkill.trim())) {
                            this.skills.push(this.newSkill.trim());
                            this.newSkill = '';
                        }
                    },
                    removeSkill(index) {
                        this.skills.splice(index, 1);
                    }
                }">
                    <label class="block text-sm font-medium text-zinc-700 dark:text-zinc-300 mb-1">
                        {{ __('frontend.skills') }}
                    </label>

                    <div class="flex gap-2 mb-2">
                        <input type="text" x-model="newSkill"
                            class="flex-1 px-4 py-2 rounded-lg border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800 text-zinc-900 dark:text-white focus:ring-2 focus:ring-indigo-500 outline-none transition"
                            placeholder="{{ __('frontend.skills_placeholder') }}"
                            @keydown.enter.prevent="addSkill" />
                        <button type="button" @click="addSkill"
                            class="px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition">
                            +
                        </button>
                    </div>

                    <!-- Skills List -->
                    <div class="flex flex-wrap gap-2">
                        <template x-for="(skill, index) in skills" :key="index">
                            <div
                                class="flex items-center bg-indigo-100 text-indigo-800 px-3 py-1 rounded-full text-sm font-medium">
                                <span x-text="skill"></span>
                                <button type="button" @click="removeSkill(index)"
                                    class="ml-2 text-indigo-500 hover:text-indigo-700">&times;</button>
                            </div>
                        </template>
                    </div>
                </div>

                <!-- ID Card Images Upload -->
                <!-- Drag & Drop + Input file -->
                <div x-data="{
                    dragging: false,
                    handleDrop(e) {
                        this.dragging = false;

                        let files = Array.from(e.dataTransfer.files);

                        // إرسال الملفات يدويًا إلى Livewire
                        $wire.uploadMultiple('id_card_images', files,
                            () => {}, // success
                            () => {} // error
                        );
                    }
                }" x-on:dragover.prevent="dragging = true"
                    x-on:dragleave.prevent="dragging = false" x-on:drop.prevent="handleDrop($event)"
                    class="relative border-2 border-dashed rounded-lg p-6 flex flex-col items-center justify-center transition duration-300"
                    :class="{
                        'border-indigo-500 bg-indigo-50': dragging,
                        'border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800':
                            !dragging
                    }">

                    <!-- Label -->
                    <div class="mb-2">
                        <label class="block text-sm font-semibold text-zinc-700 dark:text-zinc-300">
                            {{ __('frontend.attach_documents') }}
                        </label>
                        <p class="text-xs text-zinc-500 dark:text-zinc-400 mt-1">
                            {{ __('frontend.required_documents_hint') }}
                        </p>
                        <ul class="list-disc list-inside text-xs text-zinc-500 dark:text-zinc-400 mt-1 space-y-1">
                            <li>{{ __('frontend.document_id_card') }}</li>
                            <li>{{ __('frontend.document_national_id') }}</li>
                            <li>{{ __('frontend.document_work_experience_certificates') }}</li>
                        </ul>
                    </div>

                    <p class="text-sm text-zinc-500 dark:text-zinc-400 mb-2 text-center">
                        {{ __('frontend.drag_here_or') }}
                    </p>

                    <label for="id_card_images"
                        class="cursor-pointer bg-indigo-600 text-white text-sm px-4 py-2 rounded-md hover:bg-indigo-700 transition">
                        {{ __('frontend.choose_files') }}
                        <input type="file" id="id_card_images" multiple wire:model.live="id_card_images"
                            class="hidden" accept="image/*,application/pdf" />

                    </label>

                </div>

                @if (!empty($id_card_images))
                    @foreach ($id_card_images as $index => $file)
                        <div
                            class="relative group w-[120px] h-[120px] flex flex-col items-center justify-center border border-zinc-300 dark:border-zinc-600 rounded-lg bg-white dark:bg-zinc-800 shadow-sm overflow-hidden">

                            <!-- زر الحذف -->
                            <button wire:click.prevent="removeImage({{ $index }})"
                                class="absolute top-1 right-1 bg-red-600 hover:bg-red-700 text-white text-xs w-5 h-5 rounded-full shadow-md z-10">
                                ×
                            </button>

                            @if (Str::endsWith($file->getClientOriginalName(), '.pdf'))
                                <div class="flex flex-col items-center justify-center px-2 text-center">
                                    <div class="text-3xl">📄</div>
                                    <span class="text-[10px] text-zinc-600 dark:text-zinc-300 truncate w-[90%]">
                                        {{ Str::limit($file->getClientOriginalName(), 14) }}
                                    </span>
                                </div>
                            @else
                                <div
                                    class="w-[100px] h-[100px] overflow-hidden rounded-md border border-zinc-300 dark:border-zinc-600 bg-white dark:bg-zinc-800">
                                    <img src="{{ $file->temporaryUrl() }}" class="w-full h-full object-cover" />
                                </div>
                                <span class="text-[10px] text-zinc-600 dark:text-zinc-300 truncate w-[90%] mt-1">
                                    {{ Str::limit($file->getClientOriginalName(), 14) }}
                                </span>
                            @endif
                        </div>
                    @endforeach
                @endif






                <!-- Agree & open modal -->
                <label class="inline-flex items-center cursor-pointer group" @click.prevent="showTerms=true">
                    <input type="checkbox" class="sr-only" wire:model.defer="agreed">
                    <div class="w-5 h-5 border-2 border-zinc-400 dark:border-zinc-600 bg-white dark:bg-zinc-800 rounded-md flex items-center justify-center transition-colors duration-300"
                        :class="{ 'bg-indigo-500 border-indigo-500': $wire.agreed }">
                        <svg class="w-3 h-3 transition-opacity duration-300 text-zinc-800 dark:text-white"
                            :class="{ 'opacity-100': $wire.agreed, 'opacity-0': !$wire.agreed }"
                            xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"
                            stroke="currentColor" stroke-width="3">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7" />
                        </svg>
                    </div>
                    <span
                        class="ml-3 text-sm text-zinc-700 dark:text-zinc-300 select-none transition-colors duration-300">{{ __('frontend.agree') }}
                        <span class="underline text-blue-600">{{ __('frontend.terms') }}</span></span>
                </label>


                <div class="flex justify-between mt-6 space-x-2 rtl:space-x-reverse">
                    <!-- Previous Button -->
                    <button type="button" wire:click="previousStep" x-show="$wire.currentStep > 1"
                        x-transition.opacity.duration.300ms x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-gray-200 dark:bg-zinc-700 text-gray-700 dark:text-white font-medium rounded-lg hover:bg-gray-300 dark:hover:bg-zinc-600 transition-all duration-300">
                        ← {{ __('frontend.previous') }}
                    </button>



                    <!-- Submit Button -->
                    <button type="submit" x-show="$wire.currentStep === 4" x-transition.opacity.duration.300ms
                        x-transition.scale.duration.300ms
                        class="px-5 py-2 bg-green-600 text-white font-medium rounded-lg hover:bg-green-700 transition-all duration-300">
                        🚀 {{ __('frontend.submit') }}
                    </button>
                </div>
                {{-- <!-- Submit -->
                <div class="flex justify-center mt-6">
                    <button type="submit" class="neon-btn delay-[1300ms] neon-green"><span
                            class="btn-text">{{ __('frontend.cv_submitbtn') }}</span><span
                            class="glow"></span></button>
                </div> --}}

            </div>
        </form>


    </div>

    <!-- ======= Modal ======= -->
    <div x-show="showTerms" x-transition.opacity
        class="fixed inset-0 bg-black/60 backdrop-blur-sm flex items-center justify-center z-50" x-cloak>
        <div x-transition.scale
            class="bg-white dark:bg-zinc-900 rounded-xl shadow-lg w-[90%] max-w-lg p-6 space-y-4 border border-zinc-200 dark:border-zinc-700 overflow-y-auto max-h-[80vh]">
            <div class="flex flex-col items-center justify-center text-center space-y-2"><img
                    src="/assets/img/logo.png" class="h-12 w-auto" alt="logo">
                <h3 class="text-lg font-semibold text-zinc-800 dark:text-white">{{ __('frontend.terms') }}</h3>
            </div>
            <ul class="list-disc ps-6 text-sm text-zinc-700 dark:text-zinc-300 space-y-1">
                <li>{!! __('frontend.term_item1') !!}</li>
                <li>{!! __('frontend.term_item2') !!}</li>
                <li>{!! __('frontend.term_item3') !!}</li>
                <li>{!! __('frontend.term_item4') !!}</li>
                <li>{!! __('frontend.term_item5') !!}</li>
            </ul>
            <div class="flex justify-end gap-3 pt-4">
                <button @click="$wire.set('agreed', false); showTerms=false"
                    class="px-4 py-2 rounded-lg text-sm bg-zinc-200 dark:bg-zinc-700 hover:bg-zinc-300 dark:hover:bg-zinc-600">{{ __('frontend.cancel') }}</button>
                <button @click="$wire.set('agreed', true); showTerms=false"
                    class="px-4 py-2 rounded-lg text-sm bg-indigo-600 text-white hover:bg-indigo-700">{{ __('frontend.agree') }}</button>
            </div>
            <a href="{{ route('terms.print') }}" target="_blank"
                class="block text-sm text-blue-600 hover:underline text-center mt-4">{{ __('frontend.view_full_terms') }}</a>
        </div>
    </div>
</div>
